
;; read a cv (eval ctx function and lookup)
;; returns reference to associated (or default) value
(define (cv-val cv)
  (let ((key (cv-evalctx cv)))
    (let ((ref (val-at key (cv-vals cv))))
      (if ref
          ref
          (make-val-ref (cv-default cv))))))

;; evaluates the ctx function of a cv
(define (cv-evalctx cv)
  (let ((thunk (strict (cv-ctx cv))))
    (strict (eval-body (closure-body thunk) (closure-env thunk)))))

;; adds/updates value of a cv
(define (cv-update! cv nval)
  (let ((key (cv-evalctx cv)))
    (let ((ref (val-at key (cv-vals cv))))    
      (if (not ref) 
          (cv-new-binding! cv key nval) ;; extend mapping
          (if (cv? (deref ref))
              (cv-update! (deref ref) nval) ;; go change inside the cv
              (setref! ref nval)))))) ;; change existing value

;; creates a new binding key->nval in cv and 
;; returns a reference to the value part of the binding.
(define (cv-new-binding! cv key nval)
  (let ((binding (vector key nval)))
    (set-cv-vals! cv (cons binding (cv-vals cv)))
    (make-val-ref binding)))

;; returns a reference to the value of the given binding
(define (make-val-ref binding)
  (make-ref 1 binding)) 
      
;; deep copy a contextual value structure
;; only the contextual value structure is copied, not the contained
;; elements (values, ctx function)           
(define (cv-structure-copy val)
  (if (not (cv? val)) val ;; not a cv -> no copy
      (make-cv (cv-structure-copy (cv-ctx val)) ;; context function may be itself contextual
               ;;list of 2-slot vectors
               (map cv-copy-val (cv-vals val))
               ;;2-slot vector
               (cv-copy-val (cv-default val)))))

(define (cv-copy-val v) ;; v is a 2-slot vector
  (vector (strict (vector-ref v 0)) ;; key maybe itself contextual
          (cv-structure-copy (vector-ref v 1)))) ;; recursively copy the value

;;; PRINTING CV
(define (cv-print cv) 
  (printf (string-append (val-tostring cv) "\n")))
  
(define (val-tostring v)
  (if (cv? v)
      (format "{vals:~a dft:~a}" (vals-tostring (cv-vals v)) (pair-tostring (cv-default v)))
      (format "~a" v)))

(define (vals-tostring vs)
  (map (lambda (ass) (pair-tostring ass)) vs))

(define (pair-tostring vs)
  (cond ((null? vs) "")
        ((not (vector? vs)) (format "~a" v))
        (else 
         (let ((a (val-tostring (vector-ref vs 0)))
               (b (val-tostring (vector-ref vs 1))))
           (string-append "[" a "->" b "]")))))

(define (cv-trace var env)
  (printf (string-append (cv-trace-ref (env-lookup var env)) "\n")))

(define (cv-trace-ref ref)
  (let ((val (deref ref)))
    (val-tostring val)))
