;;;;;;;;;;;;;;;; environments ;;;;;;;;;;;;;;;;

(define-struct environment ())
(define-struct (empty-env    environment) ())
(define-struct (extended-env environment) (ids vals env))

;; a reference is a position in a vector
(define-struct ref (pos vec))

;; return value at ref
(define deref
  (lambda (ref)
    (vector-ref (ref-vec ref) (ref-pos ref))))

;; set value for ref
(define setref!
  (lambda (ref val)
    (vector-set! (ref-vec ref) (ref-pos ref) val)
    #t))

;; new environment with new mappings
(define new-env
  (lambda (ids vals)
    (extend-env ids vals (make-empty-env)))) 

;; extend environment with new mappings
(define extend-env
  (lambda (ids vals env)
    (make-extended-env ids (list->vector vals) env)))

;; lookup id in environment, return ref
(define env-lookup
  (lambda (id env)
    (cond ((empty-env? env) (error 'env-lookup "No binding for ~s" id))
          ((extended-env? env)
           (let ((pos (find-position id (extended-env-ids env))))
             (if (number? pos)
                 (make-ref pos (extended-env-vals env))
                 (env-lookup id (extended-env-env env)))))
          (else (error 'env-lookup "not an environment -- ~s" env)))))

;; HELPER FUNCTIONS

(define find-if
  (lambda (pred lst)
    (cond ((null? lst)      #f)
	  ((pred (car lst)) (car lst))
	  (else             (find-if pred (cdr lst))))))

(define collect-if
  (lambda (pred lst)
    (cond ((null? lst)      '())
	  ((pred (car lst)) (cons (car lst) (collect-if pred (cdr lst))))
	  (else             (collect-if pred (cdr lst))))))

(define find-position
  (lambda (sym lst)
    (let loop ((tail lst) (pos 0))
      (cond ((null? tail) #f)
	    ((eqv? sym (car tail)) pos)
	    (else (loop (cdr tail) (+ pos 1)))))))

(define find-last-position
  (lambda (sym lst)
    (let loop
	((tail lst) (curpos 0) (lastpos #f))
      (cond
       ((null? tail) lastpos)
       ((eqv? sym (car tail))
	(loop (cdr tail) (+ curpos 1) curpos))
       (else (loop (cdr tail) (+ curpos 1) lastpos))))))

;; find vector with key in list of vects, returns ref to associated value
(define (val-at key lst)
  (let ((res (find-if (lambda (vec) (eq? (vector-ref vec 0) key)) lst)))
    (if res
        (make-ref 1 res)
        res)))