(load "env.scm")
(load "parser.scm")
(load "cv.scm")

(define-struct expression ())
(define-struct (lit-exp   expression) (value))
(define-struct (var-exp   expression) (name))
(define-struct (if-exp    expression) (test then else))
(define-struct (let-exp   expression) (params args body))
(define-struct (fun-exp   expression) (params body))
(define-struct (app-exp   expression) (fun args))
(define-struct (prim-exp  expression) (prim args))
(define-struct (set!-exp  expression) (name nval))
;; make a contextual value
(define-struct (make-cv-exp expression) (ctx init))
;; display internal structure of a cv (debugging!)
(define-struct (cv-trace-exp expression) (sym))

;; values
(define-struct value ())
(define-struct (closure value) (params body env))
(define-struct (cv      value) (ctx vals default)) ;; vals is a list of 2-slot vectors
                                                   ;; default is a 2-slot vector

;; top-level evaluation
(define (eval-prog prog)
  (let ((expr (parse prog)))
    (strict (eval expr (make-empty-env)))))

;; evaluate expression exp in lexical environment env
(define (eval exp env)
  (cond ((lit-exp? exp) (lit-exp-value exp))
        
        ((prim-exp? exp)
         (let ((args (strict-eval-args (prim-exp-args exp) env)))
           (apply (prim-exp-prim exp) args)))
        
        ((if-exp? exp)
         (if (strict (eval (if-exp-test exp) env))
             (eval (if-exp-then exp) env)
             (eval (if-exp-else exp) env)))
         
        ((let-exp? exp)
         (let ((args   (eval-args (let-exp-args exp) env)))
           (eval-body (let-exp-body exp) (extend-env (let-exp-params exp) (by-val args) env))))
        
        ((make-cv-exp? exp)
         (make-cv (eval (make-cv-exp-ctx exp) env) '() 
                  (vector #f (eval (make-cv-exp-init exp) env))))
        
        ((cv-trace-exp? exp)
         (let ((id (cv-trace-exp-sym exp)))
           (cv-trace id env)
           ))
        
        ((var-exp? exp)  (deref (env-lookup (var-exp-name exp) env)))
                                   
        ((set!-exp? exp) (setval! (env-lookup (set!-exp-name exp) env) 
                                  (eval (set!-exp-nval exp) env)))
        
        ((fun-exp? exp)  (make-closure (fun-exp-params exp) (fun-exp-body exp) env))
        
        ((app-exp? exp)   (let* ((cl   (strict (eval (app-exp-fun exp) env)))
                                 (args (eval-args (app-exp-args exp) env))
                                 (env  (extend-env (closure-params cl) (by-val args) (closure-env cl))))
                            (eval-body (closure-body cl) env)))))
        
(define (eval-body body env)
    (let loop ((exps body) (val #f))
      (if (null? exps)
          val
          (loop (cdr exps) (eval (car exps) env)))))

(define (eval-args args env)
    (map (lambda (x) (eval x env)) args))

;; strictly evaluates list of argument expressions
;; (used for primitive applications)
(define (strict-eval-args args env)
    (map (lambda (x) (strict (eval x env))) args))

;; reduces a value to an actual (non-contextual) value
(define (strict val)
  (if (cv? val) (strict (deref (cv-val val)))
      val))

;; ensures the list of arguments is passed with the by-value semantics:
;; contextual values should not imply shared state, so all contextual values
;; are copied (only the structure, not the non-contextual values)
(define (by-val l)
  (map cv-structure-copy l))

;; set a reference: if r points to a normal value,
;; update it, otherwise update the contextual value.
(define (setval! r nval)
  (let ((val (deref r)))
    (if (cv? val)
        (cv-update! val nval)
        (setref! r nval))))
