(load "env.scm")
(load "ast.scm")
(load "parser.scm")


;; INTERPRETER FOR BASE

(define run
  (lambda (text)
    (eval-program (parse-program text))))

(define eval-program
  (lambda (pgm)
    (set! *classes* (init-classes))
    (elaborate-class-decls! (program-decls pgm))
    (eval
      (parse-exp `(send (new Main) main))
      (make-empty-env)
      '()
      root-jp)))

;;;;;;;;;;;;;;; runtime structures ;;;;;;;;;;;;;;;;;

; objects
(define-struct object (class vals aspects))

; deployed aspects 
(define-struct dasp (c d f pc adv))

; join points
(define-struct jp   (kind name target args parent))
(define root-jp (make-jp #f #f #f '() #f))

; closures (used for pointcuts) 
(define-struct closure (params body env aspects))

; classes
(define-struct class  (name super fields methods))
(define-struct method (class name params body))
(define-struct field  (class type name))
(define *classes* '())

;; build object, int and string classes
(define init-classes
  (lambda () 
    (let ((class-object (make-class 'Object #f '() (list (make-method 'Object 'init '()
                                                           (parse-body '(1))))))
          (class-int    (make-class 'int    #f '() '()))
          (class-string (make-class 'string #f '() '())))
      (list class-object class-int class-string))))


(define lookup-class 
  (lambda (cname)
    (or (find-if (lambda (class) (eqv? (class-name class) cname)) *classes*)
        (error 'lookup-class "No class named ~s." cname))))

(define elaborate-class-decls!
  (lambda (decls)
    (for-each (lambda (decl)
		(let* ((cname (class-decl-cname decl))
		       (super (lookup-class (class-decl-sname decl)))
		       (fdecls (collect-if  field-decl? (class-decl-decls decl)))
		       (mdecls (collect-if method-decl? (class-decl-decls decl)))
                       (class (make-class cname super '() '()))
		       (fields  (get-fields class fdecls))
		       (methods (make-methods class mdecls)))
                  (set-class-fields! class fields)
                  (set-class-methods! class methods)
		  (set! *classes* (append *classes* (list class)))
		  class))
	      decls)))

(define get-fields
  (lambda (class fdecls)
    (let ((super-fields (class-fields (class-super class)))		
	  (fields (make-fields class fdecls)))
      (append super-fields fields))))

(define make-fields
  (lambda (class fdecls)
    (map (lambda (fdecl) (make-field class
                                     (field-decl-type fdecl)
                                     (field-decl-fname fdecl)))
         fdecls)))

(define make-methods
  (lambda (class mdecls)
    (map (lambda (mdecl) (make-method class 
                                      (method-decl-mname mdecl)
                                      (method-decl-params mdecl)
                                      (method-decl-body mdecl)))
         mdecls)))

;;;;;;;;;;;;;;;; the interpreter ;;;;;;;;;;;;;;;;

(define (eval exp E A jp)
  (cond ((lit-exp? exp) (lit-exp-value exp))
        
        ((prim-app-exp? exp)
         (let ((args (eval-args (prim-app-exp-args exp) E A jp)))
           (apply (prim-app-exp-prim exp) args)))
        
        ((if-exp? exp)
         (if (eval (if-exp-test exp) E A jp)
             (eval (if-exp-then exp) E A jp)
             (eval (if-exp-else exp) E A jp)))
        
        ((and-exp? exp)
         (let loop ((args (and-exp-args exp)))
           (if (null? args)
               #t
               (and (eval (car args) E A jp)
                    (loop (cdr args))))))
        
        ((or-exp? exp)
         (let loop ((args (or-exp-args exp)))
           (if (null? args)
               #f
               (or (eval (car args) E A jp)
                   (loop (cdr args))))))
        
        ((let-exp? exp)
         (let ((pnames (pnames (let-exp-params exp)))
               (args   (eval-args (let-exp-args exp) E A jp)))
           (eval-body (let-exp-body exp) (extend-env pnames args E) A jp)))
        
        ((vget? exp)
         (let ((name  (vget-name exp)))
           (env-get name E)))
        
        ((vset? exp)
         (let ((name (vset-name exp))
               (nval  (eval (vset-val exp) E A jp)))
           (env-set! nval name E)
           'undefined))
        
        ((call? exp)
         (let* ((sig (call-sig exp))
                (obj (eval (call-target exp) E A jp))
                (args (eval-args (call-args exp) E A jp)))
           (call-method sig obj args A jp))) 
        
        ((scall? exp)
         (let* ((sig  (scall-sig exp))
                (args  (scall-args exp))
                (class (env-get '%host E))
                (obj    (env-get 'this E))
                (args   (eval-args args E A jp)))
           (call-super sig (class-super class) obj args A jp)))
        
        ((new? exp)
         (new-object (lookup-class (new-class exp))
                     (eval-args (new-args exp) E A jp) A jp))
        
        ((fget? exp)
         (let* ((obj  (env-get 'this E))
                (name (fget-name exp)))
           (get-field-value name (env-get '%host E) (object-vals obj))))
        
        ((fset? exp)
         (let* ((obj  (env-get 'this E))
                (name (fset-name exp))
                (nval (eval (fset-val exp) E A jp)))
           (set-field-value! name (env-get '%host E) (object-vals obj) nval)
           'undefined))
        
        ((instanceof? exp)
         (is-instanceof? (eval (instanceof-target exp) E A jp)
                         (lookup-class (string->symbol (instanceof-class exp)))))
        
        ;; aspects
        ((this-jp-exp? exp) jp)
        
        ((depl-exp? exp)  (let ((dasp (build-dasp exp E A jp)))
                            (eval-body (depl-exp-body exp) E (cons dasp A) jp)))
        
        ;; functions (just for pointcuts)
        ((fun-exp? exp)   (make-closure (fun-exp-params exp) (fun-exp-body exp) E '()))
        
        (else (error 'eval "not an expression -- ~s" exp))))

(define eval-body
  (lambda (body env asps jp)
    (let loop ((exps (body-exps body)) (val #f))
      (if (null? exps)
          val
          (loop (cdr exps) (eval (car exps) env asps jp))))))

(define eval-args
  (lambda (args env asps jp)
    (map (lambda (x) (eval x env asps jp)) args)))

(define call-method
  (lambda (sig obj args asps jp)
;    (printf "---> call ~a with A(~a)~n" sig (length asps))
    (let ((njp (make-jp 'call sig obj args jp)))
      (weave-some asps njp)
      (execute (lookup-method sig (object-class obj)) 
               obj args asps njp))))

(define call-super
  (lambda (sig scls obj args asps jp)
    (execute (lookup-method sig scls) obj args asps jp)))

(define execute
  (lambda (method this args asps jp)
    (let* ((env (new-env (append '(this %host) (pnames (method-params method)))
                         (append (list this (method-class method)) args)))
           (nasps (union (collect-match-c jp asps) 
                         (object-aspects this)))
           (njp (make-jp 'exec (method-name method) this args jp)))
      (weave-some nasps njp)
      (eval-body (method-body method) env nasps njp)))) 

(define lookup-method
  (lambda (sig class)
    (if (not class) ;; reached top
        (error 'eval "No method ~s" sig)
        (let ((method (find-if (lambda (method) (eqv? (method-name method) sig))
                               (class-methods class))))
          (if (method? method)
              method
              (lookup-method sig (class-super class)))))))

(define new-object
  (lambda (class args asps jp)
    (let ((njp (make-jp 'new #f class args jp)))
      (weave-some asps njp)
      (let* ((fields (class-fields class))
             (vals   (make-vector (length fields)))
             (nasps  (collect-match-d njp asps))
             (obj    (make-object class vals nasps)))
        ;(printf "-- new object ~a with A(~a)~n" (class-name class) (length nasps))
        (call-method 'init obj args asps njp)
        obj))))
    
(define get-field-value
  (lambda (fname class vals)
    (let ((pos (lookup-field-pos fname class)))
      (vector-ref vals pos))))
    
(define set-field-value! 
  (lambda (fname class vals nval)
    (let ((pos (lookup-field-pos fname class)))
      (vector-set! vals pos nval))))

(define lookup-field-pos
  (lambda (fname class)
    (let ((pos (find-last-position fname (fnames (class-fields class)))))
      (if (number? pos) 
          pos
          (error 'lookup-field
                 "~s does not have a field ~s." (class-name class) fname)))))

(define fnames (lambda (fields) (map field-name fields)))



;; aspect environments
(define (collect-match-c jp asps) (collect-if (lambda (a) (app/prim (dasp-c a) jp)) asps))
(define (collect-match-d jp asps) (collect-if (lambda (a) (app/prim (dasp-d a) jp)) asps))
(define (collect-match-f jp asps) (collect-if (lambda (a) (app/prim (dasp-f a) jp)) asps))


;; deployed aspects
(define (build-dasp exp E A jp)
  (let ((ds  (eval (depl-exp-ds exp) E A jp))
        (asp (eval (depl-exp-asp exp) E A jp)))
    (make-dasp (car ds)
               (cadr ds)
               (caddr ds)
               (car asp)
               (cadr asp))))

;; weaving
(define (weave-some asps jp)
  (weave-all (collect-match-f jp asps) jp))
   
(define (weave-all asps jp)
  (map (lambda (x) (weave x jp)) asps))

(define (weave asp jp)
  (if (app/prim (dasp-pc asp) jp)
      (app/prim (dasp-adv asp) jp)))

;; applies the (pointcut) function f, passing jp as argument
;; evaluation is done in an empty aspect environment
(define (app/prim f jp)
  (eval-body (closure-body f)
             (extend-env (closure-params f) (list jp)
                         (closure-env f)) '() jp))


  